/**
* \file: partition_list.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>

#include "utils/logger.h"

#include "model/partition_list.h"
#include "model/device_list.h"
#include "model/model_internal.h"

static partition_t* partition_find_next_mounted_partition(partition_t *partition,
		bool iterate_through_devices);

partition_t *device_add_partition_by_device(device_t *device, const char *partition_id,
		 partition_metadata_t *metadata)
{
	partition_t *first_partition, *partition;
	partition = partition_create_new(partition_id, device, metadata);

	if (partition!=NULL)
	{
		//put in the new partition between head and the first partition
		first_partition = device->partition_list_head.next_partition;

		//update forward links
		device->partition_list_head.next_partition=partition;
		partition->next_partition=first_partition;

		//update backwards links
		partition->previous_partition=&device->partition_list_head;
		if (first_partition!=NULL)
			first_partition->previous_partition=partition;

		device->detected_partition_cnt++;
	}
	else
		logger_log_error("PARTITION_LIST - Memory problems while adding a partition %s to the device.",partition_id);

	return partition;
}

partition_t *device_add_partition_by_device_id(const char *device_id, const char *partition_id,
		partition_metadata_t *metadata)
{
	device_t *device;

	device=device_list_find_device(device_id);
	if (device!=NULL)
		return device_add_partition_by_device(device, partition_id, metadata);
	else
		return NULL;
}

void device_delete_partition(partition_t *partition)
{
	partition_t *par_before_partition, *par_after_partition;

	partition->device->detected_partition_cnt--;

	par_before_partition=partition->previous_partition;
	par_after_partition=partition->next_partition;

	//update the forward link
	par_before_partition->next_partition=par_after_partition;

	//update the backward link
	if (par_after_partition!=NULL)
		par_after_partition->previous_partition=par_before_partition;

	//delete the device
	partition_free(partition);
}



partition_t *device_find_partition(device_t *device, const char *partition_id)
{
	partition_t *partition;
	partition=device->partition_list_head.next_partition;

	while (partition != NULL)
	{
		if (strcmp(partition->id,partition_id)==0)
			return partition;
		partition=partition->next_partition;
	}

	//not found
	return NULL;
}

partition_t* device_list_find_partition_by_id(const char *partition_id)
{
	device_list_iterator_t dev_iter;
	device_t *device;
	partition_t *partition=NULL;

	if (partition_id==NULL) return NULL;

	device=device_list_first_element(&dev_iter);
	while(device != NULL && partition==NULL)
	{
		partition=device_find_partition(device,partition_id);
		device=device_list_next_element(&dev_iter);
	}

	return partition;
}

partition_t* device_list_find_partition_by_mountpoint(const char *mount_point)
{
	device_list_iterator_t dev_iterator;
	partition_iterator_t part_iterator;

	device_t *device;
	partition_t *partition;

	if (mount_point==NULL)
		return NULL;

	device=device_list_first_element(&dev_iterator);
	while(device != NULL)
	{
		partition=device_first_partition(device,&part_iterator);
		while(partition != NULL)
		{
			if (partition->mount_point != NULL)
			{
				if (strcmp(partition->mount_point, mount_point)==0) return partition;
			}
			partition=device_next_partition(&part_iterator);
		}

		device=device_list_next_element(&dev_iterator);
	}

	return NULL;
}



partition_t *device_first_partition(device_t *device, partition_iterator_t *iterator)
{
	if (iterator != NULL)
		(*iterator)=(partition_iterator_t)device->partition_list_head.next_partition;

	return device->partition_list_head.next_partition;
}

partition_t *device_next_partition(partition_iterator_t *iterator)
{
	partition_t *current_element=(partition_t *)(*iterator);
	if (current_element!=NULL)
	{
		(*iterator)=(partition_iterator_t)current_element->next_partition;
		return current_element->next_partition;
	}
	else
		return NULL;
}



static partition_t* partition_find_next_mounted_partition(partition_t *partition,
		bool iterate_through_devices)
{
	device_t *device;

	if (partition==NULL) return NULL;

	device=partition->device;
	partition=partition->next_partition;

	while (device!=NULL)
	{
		while (partition!=NULL)
		{
			if (partition->state == PARTITION_MOUNTED)
				return partition;
			partition=partition->next_partition;
		}

		if (!iterate_through_devices) break;

		device=device->next_device;
		if (device!=NULL)
			partition=device->partition_list_head.next_partition;
	}

	return NULL;
}

partition_t *device_first_mounted_partition(device_t *device, partition_iterator_t *iterator)
{
	partition_t *partition;
	partition=partition_find_next_mounted_partition(&device->partition_list_head,false);

	if (iterator != NULL)
		(*iterator)=(partition_iterator_t)partition;

	return partition;
}

partition_t *device_next_mounted_partition(partition_iterator_t *iterator)
{
	partition_t *partition;
	partition=(partition_t *)(*iterator);
	partition=partition_find_next_mounted_partition(partition,false);

	(*iterator)=(partition_iterator_t)partition;

	return partition;
}

partition_t *device_list_first_mounted_partition(partition_iterator_t *iterator)
{
	partition_t *partition;
	device_t *device;

	device=device_list_first_element(NULL);
	if (device==NULL) return NULL;

	partition=partition_find_next_mounted_partition(&device->partition_list_head,true);

	if (iterator != NULL)
		(*iterator)=(partition_iterator_t)partition;

	return partition;
}

partition_t *device_list_next_mounted_partition(partition_iterator_t *iterator)
{  
	partition_t *partition;

	partition=(partition_t *)(*iterator);
	partition=partition_find_next_mounted_partition(partition,true);

	(*iterator)=(partition_iterator_t)partition;

	return partition;
}
